# -*- coding: utf-8 -*-
"""
Created on Mar 29, 2016

@author: awhittington
@copyright: 2016 Southwest Research Institute ®

"""

import csv
import argparse
from collections import defaultdict
from collections import OrderedDict
import itertools

from lxml import etree

ns = {"xsd": "http://www.w3.org/2001/XMLSchema",
      "mdl": "http://www.wsmr.army.mil/RCC/schemas/MDL"
}
interfaces = ['BaseIF', 'CreationIF', 'NegotiationIF', 'ManagementIF', 'NetworkNodeIF', 'DeviceIF', 'ManagerIF',
              'LTCDataSourceIF', 'LTCDataSinkIF', 'RCDataSourceIF', 'RCDataSinkIF', 'DataAcquisitionIF', 'RecordingIF',
              'NetworkFabricIF', 'TimeSourceIF', 'VoiceIF', 'RFTransceiverIF', 'RFLinkConfigIF', 'RFLinkControlIF',
              'TransceiverTSSIF', 'RFLinkManagementTSSIF', 'SSTSourceIF', 'SSTSinkIF', 'AntennaControlIF', 'EncryptionIF'
]
components = {'DAU': ['BaseIF', 'NegotiationIF', 'ManagementIF', 'NetworkNodeIF', 'DeviceIF', 'LTCDataSourceIF', 'DataAcquisitionIF'],
              'Recorder': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'LTCDataSourceIF', 'LTCDataSinkIF', 'RCDataSourceIF', 'RecordingIF'],
              'Switch': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'NetworkFabricIF', 'TimeSourceIF'],
              'Radio': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'RFTransceiverIF', 'RFLinkConfigIF', 'TransceiverTSSIF'],
              'LinkManager': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'RFLinkConfig', 'RFLinkControlIF', 'RFLinkManagementTSSIF'],
              'NetworkGateway': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'LTCDataSinkIF'],
              'PCMGateway': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'LTCDataSourceIF'],
              'VoiceGateway': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'VoiceIF'],
              'SystemManager': ['CreationIF', 'NegotiationIF'],
              'Encryptor': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'EncryptionIF'],
              'SSTTx': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'SSTSourceIF'],
              'SSTRx': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'SSTSinkIF'],
              'ACU': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'AntennaControlIF'],
              'OnboardTAManager': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'ManagerIF'],
              'Ground': ['BaseIF', 'ManagementIF', 'NetworkNodeIF', 'RCDataSinkIF']
}
ch22 = OrderedDict([
    ("22.1 General", []),
    ("22.1.1 General NetworkNode Requirements", ["NetworkNodeIF"]),
    ("22.1.2 General NetworkDevice Requirements", ["NetworkFabricIF"]),
    ("22.2 Network Access Layer", []),
    ("22.2.1 Physical Layer", []),
    ("22.2.1.1 Wired Ethernet", []),
    ("22.2.1.2 Wireless Technologies", []),
    ("22.2.2 Data Link Layer Protocols", ["NetworkNodeIF"]),
    ("22.2.2.1 Frame Structure", ["NetworkNodeIF"]),
    ("22.2.2.2 Media Access Control (MAC)", ["NetworkNodeIF"]),
    ("22.2.2.3 Logical Link Control (LLC)", ["NetworkNodeIF"]),
    ("22.2.2.4 Link Layer Switching", ["NetworkFabricIF"]),
    ("22.2.2.5 Link Layer Bridging", ["NetworkFabricIF"]),
    ("22.2.2.6 Link Layer Flow Control", ["NetworkNodeIF"]),
    ("22.2.2.7 Address Resolution", ["NetworkNodeIF"]),
    ("22.3 Internet Layer Protocols", ["NetworkNodeIF"]),
    ("22.3.1 Internet Protocol version 4 (IPv4)", ["NetworkNodeIF"]),
    ("22.3.1.1 Internet Control Message Protocol (ICMP)", ["NetworkNodeIF", "NetworkFabricIF"]),
    ("22.3.1.2 Internet Group Management Protocol (IGMP)", ["NetworkNodeIF"]),
    ("22.3.2 Internet Protocol version 6 (IPv6)", ["NetworkNodeIF"]),
    ("22.3.2.1 Internet Control Message Protocol Version 6 (ICMPv6)", ["NetworkNodeIF"]),
    ("22.3.2.2 Multicast Listener Discovery (MLD) for IPv6", ["NetworkFabricIF"]),
    ("22.3.3 IP Datagram Transmission", ["NetworkNodeIF"]),
    ("22.3.4 Protocol Independent Multicast (PIM)", ["NetworkFabricIF"]),
    ("22.3.5 Network Routing", ["NetworkNodeIF", "NetworkFabricIF"]),
    ("22.4 Transport Layer Protocols", ["NetworkNodeIF"]),
    ("22.4.1 Transmission Control Protocol (TCP)", ["NetworkNodeIF"]),
    ("22.4.2 User Datagram Protocol (UDP)", ["NetworkNodeIF"]),
    ("22.4.3 Transport Layer Security (TLS) and Secure Sockets Layer (SSL)", ["NetworkNodeIF"]),
    ("22.5 Application Layer Protocols", ["NetworkNodeIF"]),
    ("22.5.1 Core NetworkNode Protocols", []),
    ("22.5.1.1 Host/Address Configuration", ["NetworkNodeIF"]),
    ("22.5.1.2 Domain Name Services", ["NetworkNodeIF"]),
    ("22.5.1.3 Time Synchronization", ["NetworkNodeIF", "NetworkFabricIF", "TimeSourceIF"]),
    ("22.5.1.4 Information Assurance and Encryption", ["NetworkNodeIF", "EncryptionIF"]),
    ("22.5.2 Core TmNSApp Protocols", ["ManagementIF"]),
    ("22.5.2.1 Simple Network Management Protocol (SNMP)", ["ManagementIF"]),
    ("22.5.2.2 Hypertext Transfer Protocol (HTTP)", ["ManagementIF"]),
    ("22.5.2.3 Real Time Streaming Protocol (RTSP)", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("22.5.2.4 File Transfer", ["ManagementIF"]),
    ("22.5.2.5 Voice Over IP", ["VoiceIF"]),
    ("22.5.2.6 Secure Communications", ["NetworkNodeIF"]),
    ("22.5.2.7 Uniform Resource Identifier (URI)/Uniform Resource Name (URN)", []),
    ("22.5.3 Quality of Service (QoS)", ["ManagementIF", "NetworkNodeIF", "LTCDataSourceIF", "RCDataSourceIF", "RFLinkConfigIF", "TransceiverTSSIF", "RFLinkManagementTSSIF"]),
    ("22.5.3.1 Differentiated Services (DiffServ)", ["ManagementIF", "NetworkNodeIF", "LTCDataSourceIF", "RCDataSourceIF", "RFLinkConfigIF", "TransceiverTSSIF", "RFLinkManagementTSSIF"]),
    ("22.5.3.2 DiffServ Code Point (DSCP) Assignments", ["ManagementIF", "NetworkNodeIF", "LTCDataSourceIF", "RCDataSourceIF", "RFLinkConfigIF", "TransceiverTSSIF", "RFLinkManagementTSSIF"])
])
ch24 = OrderedDict([
    ("24.1 General", []),
    ("24.1.1 Type-Length Value (TLV) Structure", []),
    ("24.2 TmNSMessage", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1 TmNSMessageHeader Structure", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.1 MessageVersion Field (4 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.2 OptionWordCount Field (4 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.3 Reserved Field (4 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.4 MessageType Field (4 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.5 MessageFlags Field (16 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.6 MessageDefinitionID Field (32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.7 MessageDefinitionSequenceNumber Field (32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.8 MessageLength Field (32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.9 MessageTimestamp Field (64 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.1.10 ApplicationDefinedFields Field (OptionWordCount*32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2 TmNSMessagePayload Structure", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1 Package Structure", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1 Package Header", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1 Standard PackageHeader", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1.1 PackageDefinitionID Field (32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1.2 PackageLength Field (16 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1.3 Reserved (8 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1.4 PackageStatusFlags Field (8 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.1.1.5 PackageTimeDelta Field (32 bits)", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.2.2.1.2 Package Time Measurement Scoping Rules", ["NegotiationIF", "LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("24.3 RF Network Message", ["RFLinkConfig"]),
    ("24.3.1 RF Network Message Header Structure", []),
    ("24.3.1.1 Message Length (16 bits)", []),
    ("24.3.1.2 Destination RF MAC Address (16 bits)", ["RFLinkConfig"]),
    ("24.3.1.3 Source RF MAC Address (16 bits)", ["RFLinkConfig"]),
    ("24.3.1.4 Message Sequence Number (32 bits)", []),
    ("24.3.2 RF Network Message Payload Structure", []),
    ("24.3.2.1 TxOp Assignment TLV", ["RFLinkConfigIF"]),
    ("24.3.2.2 TxOp ID Acknowledgement Report TLV", []),
    ("24.3.2.3 MAC Queue Status Report TLV", []),
    ("24.3.2.4 Heartbeat TLV", ["RFLinkConfigIF"]),
    ("24.3.2.5 Link Metric TLV", []),
    ("24.3.2.6 Traffic Engineering (TE) Queue Status Report TLV", []),
    ("24.4 TSS Messages", ["TransceiverTSSIF", "RFLinkManagementTSSIF"]),
    ("24.4.1 TSS Initialization Message Structure", []),
    ("24.4.1.1 Interface Parameter Identifier (4 Bytes)", []),
    ("24.4.1.2 Interface Parameter", []),
    ("24.4.2 TSS Data Message Structure", []),
    ("24.4.2.1 Message Length Field (16 bits)", []),
    ("24.4.2.2 Cyclic Redundancy Check (CRC)", []),
    ("24.4.2.3 Encapsulated Ethernet Frame (Variable Length)", [])
])
ch25 = OrderedDict([
    ("25.1 General", []),
    ("25.2 Structure of Management Resources", []),
    ("25.2.1 Public RFC-Based Management Resources", []),
    ("25.2.1.1 Public RFC Management Information Base (MIB) Support", []),
    ("25.2.1.2 Notifications Support", []),
    ("25.2.1.3 Table Management using the RowStatus Column", []),
    ("25.2.2 TmNS-Specific Management Resources", []),
    ("25.2.2.1 tmnsTmaCommon", ["ManagementIF"]),
    ("25.2.2.2 tmnsTmaSpecificCapabilities", ["LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF", "DataAcquisitionIF", "RecordingIF", "NetworkFabricIF", "TimeSourceIF", "VoiceIF", "RFTransceiverIF", "RFLInkConfigIF", "RFLinkControlIF", "TransceiverTSSIF", "RFLinkManagementTSSIF", "SSTSourceIF", "SSTSinkIF", "AntennaControlIF", "EncryptionIF"]),
    ("25.2.2.3 tmnsNetworkNode", ["NetworkNodeIF"]),
    ("25.2.2.4 tmnsGeneralNotification", ["ManagementIF"]),
    ("25.3 Management Resource Matrix", []),
    ("25.3.1 Hierarchy Element Class", []),
    ("25.3.2 Resource Name", []),
    ("25.3.3 Parent Resource Name", []),
    ("25.3.4 Resource Position", []),
    ("25.3.5 Resource URN", []),
    ("25.3.6 MIB OID", []),
    ("25.3.7 Resource Syntax", []),
    ("25.3.8 Access Level", []),
    ("25.3.9 Default Value", []),
    ("25.3.10 Table Index #", []),
    ("25.3.11 Data Introduced", []),
    ("25.3.12 Persistent", []),
    ("25.3.13 Idempotency", []),
    ("25.3.14 Description", []),
    ("25.3.15 Comment", []),
    ("25.4 Management Protocols", []),
    ("25.4.1 SNMP-based ManagementResources", []),
    ("25.4.2 HTTP-based ManagementResources", []),
    ("25.4.3 TmNS Resource Management Protocols", ["BaseIF", "ManagementIF", "ManagerIF"]),
    ("25.4.3.1 Device Configuration Protocol", ["BaseIF", "ManagementIF", "ManagerIF"]),
    ("25.4.3.2 File Export Protocols", ["BaseIF", "ManagementIF", "ManagerIF"]),
    ("25.4.3.3 TmNS Configuration Negotiation Protocol", ["NegotiationIF", "ManagerIF"]),
    ("25.5 Uniform Resource Name (URN)", [])
])
ch26 = OrderedDict([
    ("26.1 General", []),
    ("26.2 Data Channel Characteristics", []),
    ("26.2.1 Network Transport Characteristics", []),
    ("26.2.1.1 UDP DataChannel", ["LTCDataSourceIF", "LTCDataSinkIF"]),
    ("26.2.1.2 TCP DataChannel", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.2.2 Message List", ["LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("26.2.3 Time Range", []),
    ("26.3 Metadata-Defined Application Data Transfer", ["LTCDataSourceIF", "LTCDataSinkIF"]),
    ("26.3.1 Latency/Throughput Critical (LTC) Delivery Protocol", ["LTCDataSourceIF", "LTCDataSinkIF"]),
    ("26.3.1.1 LTC Delivery Protocol Data Channel (LTCDataChannel)", ["LTCDataSourceIF", "LTCDataSinkIF"]),
    ("26.4 Request-Defined Application Data Transfer", []),
    ("26.4.1 RTSP-Based Control Channel (RTSPControlChannel)", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.1.1 Transport Header", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.1.2 Range Header", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.1.3 Bandwidth and Speed Headers", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.1.4 Request-Defined URI Syntax", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.1.5 Request Types", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.2 RTSP-Based Data Channel (RTSPDataChannel)", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.2.1 TCP-Based RTSPDataChannel", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.2.2 End of Data Indication", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.3 Reliability Critical (RC) Delivery Protocol", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.3.1 RC Delivery Protocol Data Channel (RCDataChannel)", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.3.2 RC Delivery Protocol Control Channel (RCControlChannel)", ["RCDataSourceIF", "RCDataSinkIF"]),
    ("26.4.4 Request-Defined Data Channel", []),
    ("26.5 TmNSDataMessage Transfer Rules", ["LTCDataSourceIF", "LTCDataSinkIF", "RCDataSourceIF", "RCDataSinkIF"]),
    ("26.5.1 Sequence Numbering Convention", []),
    ("26.5.2 Timestamp Convention", []),
    ("26.5.3 TmNSDataMessage Fragmentation", []),
    ("26.5.4 Generating TmNSDataMessages from Other TmNSDataMessages Convention", []),
])
ch27 = OrderedDict([
    ("27.1 Introduction", []),
    ("27.2 Radio Access Network Concepts and Definitions", []),
    ("27.2.1 Data Link Layer Framing", []),
    ("27.2.2 RF Media Access Control Layer", []),
    ("27.2.3 Epoch Structure", ["RFLinkConfigIF"]),
    ("27.2.4 Transmission Opportunities (TxOps)", ["RFLinkConfigIF"]),
    ("27.2.5 Timing", []),
    ("27.2.6 Radio Link State Non-Volatile Storage", ["RFTransceiverIF"]),
    ("27.2.7 Frame Check Sequence (FCS)", []),
    ("27.2.8 ARQ", []),
    ("27.3 Physical Layer", []),
    ("27.3.1 Data Rates and Spectrum", []),
    ("27.3.1.1 Radio Air Data Rates", []),
    ("27.3.1.2 Spectrum", ["RFLinkConfigIF"]),
    ("27.3.2 Regulatory Specifications, Spectral Mask", []),
    ("27.3.2.1 SOQPSK-TG Single-Carrier Waveform - Spectral Mask", []),
    ("27.3.2.2 SOQPSK-TG Single-Carrier Waveform - Bandwidth", []),
    ("27.3.2.2.1 Occupied Bandwidth", []),
    ("27.3.2.2.2 Air Information Bit Rate", []),
    ("27.3.2.2.3 Guard-Bands and Band Edge Spurious Level", []),
    ("27.3.2.3 Multiple-Carrier Waveform - Spectral Mask", []),
    ("27.3.2.4 Multiple-Carrier Waveform - Bandwidth", []),
    ("27.3.3 Carrier and Clock Frequency Error, Phase Noise, Spurs, Receiver Sensitivity", []),
    ("27.3.3.1 SOQPSK-TG Single-Carrier Transmission", []),
    ("27.3.3.1.1 Carrier Frequency Error", []),
    ("27.3.3.1.2 Transceiver Phase Noise", []),
    ("27.3.3.1.3 Receiver Sensitivity", []),
    ("27.3.3.1.4 Frequency Error Attributed to Doppler", []),
    ("27.3.3.1.5 Symbol Clock Frequency Error", []),
    ("27.3.3.1.6 Transmission Time Accuracy", []),
    ("27.3.3.2 Multiple-Carrier Transmission", []),
    ("27.4 RF Burst Format", []),
    ("27.4.1 Physical Layer Modulation", []),
    ("27.4.2 Preamble", []),
    ("27.4.3 Attached Synchronization Marker (ASM)", []),
    ("27.4.4 Pseudo-Randomization", []),
    ("27.4.5 Low-Density Parity-Check (LDPC)", []),
    ("27.4.5.1 Encoder Implementation", []),
    ("27.5 RF Media Access Control Frame Structure", []),
    ("27.5.1 RF MAC Header", []),
    ("27.5.1.1 Frame Control", []),
    ("27.5.1.1.1 Reserved Field 1 (6 bits)", []),
    ("27.5.1.1.2 Version Field (2 bits)", []),
    ("27.5.1.1.3 Reserved Field 2 (6 bits)", []),
    ("27.5.1.1.4 Protected Frame Field (2 bits)", []),
    ("27.5.1.2 Destination Address (16 bits)", ["RFTransceiverIF"]),
    ("27.5.1.3 Source Address (16 bits)", ["RFTransceiverIF"]),
    ("27.5.1.4 Reserved (3 bits)", []),
    ("27.5.1.5 Length (13 bits)", []),
    ("27.5.2 CCMP Header", []),
    ("27.5.3 RF MAC Payload", []),
    ("27.5.4 Message Integrity Code", []),
    ("27.5.5 Frame Check Sequence Field", []),
    ("27.6 Power Transients", []),
    ("27.6.1 Radio RF Power Transients Within a TxOp Allocation", [])
])
ch28 = OrderedDict([
    ("28.1 Introduction", []),
    ("28.2 RF Network Management Concepts and Definitions", []),
    ("28.2.1 Data Link Layer Framing", []),
    ("28.2.2 RF Link Management", []),
    ("28.2.3 Epoch Structure", ["RFLinkConfigIF"]),
    ("28.2.4 Epoch Length", ["RFLinkConfigIF"]),
    ("28.2.5 Transmission Opportunities (TxOps)", ["RFLinkConfigIF"]),
    ("28.2.6 Guard Time", []),
    ("28.2.7 TE Queues", ["RFLinkConfigIF"]),
    ("28.2.8 Handoff", []),
    ("28.2.9 Heartbeat", ["RFLinkConfigIF"]),
    ("28.2.10 RF MAC Header Addressing", ["RFTransceiverIF"]),
    ("28.2.11 Timing", []),
    ("28.2.12 Virtual TxOps", []),
    ("28.2.13 Independent Operation", ["RFLinkConfigIF"]),
    ("28.3 RF Media Access Control Layer", []),
    ("28.3.1 Epoch", ["RFLinkConfigIF"]),
    ("28.3.1.1 Epoch Structure", ["RFLinkConfigIF"]),
    ("28.3.1.2 Epoch Timing", ["RFLinkConfigIF"]),
    ("28.3.1.3 Guard Times", []),
    ("28.3.1.4 Transmission Opportunity", ["RFLinkConfigIF"]),
    ("28.3.2 Media Access Control Frame Structure", ["RFTransceiverIF"]),
    ("28.3.2.1 Header Format", ["RFTransceiverIF"]),
    ("28.3.2.2 Unprotected Payload", []),
    ("28.3.2.3 Protected Payload", []),
    ("28.3.3 RF MAC Payloads", []),
    ("28.3.3.1 RF MAC Service Data Units", []),
    ("28.3.3.2 RF MAC Frame Fragmentation", []),
    ("28.3.4 Frame Check Sequence", []),
    ("28.3.5 ARQ Processing", []),
    ("28.4 Logical Link Control Layer", ["RFLinkConfigIF"]),
    ("28.4.1 TxOp Processing", ["RFLinkConfigIF"]),
    ("28.4.1.1 TxOp Processing After Power Interruption", ["RFTransceiverIF"]),
    ("28.4.1.2 TxOp Processing When Heartbeat Times Out", ["RFTransceiverIF"]),
    ("28.4.2 Queue Management Processing", ["RFLinkConfigIF"]),
    ("28.4.3 Link Metric Processing", []),
    ("28.4.4 Heartbeat Processing", ["RFLinkConfigIF"]),
    ("28.5 Tunnel Management", []),
    ("28.5.1 Tunnel Connection", []),
    ("28.5.2 TSS Interface Initialization", ["TransceiverTSSIF", "RFLinkManagementTSSIF"]),
    ("28.5.2.1 Initialization of Transceiver TSS Interface", ["TransceiverTSSIF"]),
    ("28.5.2.2 Initialization of RF Link Management TSS Interface", ["RFLinkManagementTSSIF"]),
    ("28.5.3 Tunnel Operation", []),
    ("28.5.4 Tunnel Selection", []),
])
chapters = {"22": ch22, "24": ch24, "25": ch25, "26": ch26, "27": ch27, "28": ch28}


def create_spreadsheet(type, schema, spreadsheet, **_):
    """
    Creates a CSV file with columns for each element contained within each type
    """
    elements = schema.xpath("//xsd:element[ancestor::xsd:complexType]", namespaces=ns)

    headers = ["Element"]
    if type == "interfaces":
        headers = headers + interfaces
    elif type == "chapters":
        headers = headers + chapters

    with open(spreadsheet, 'w', newline='') as spreadsheet:
        writer = csv.writer(spreadsheet)
        writer.writerow(headers)

        previous_complex_type = elements[0].xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
        for element in elements:
            complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
            row = complex_type + " - " + element.get('name')

            if complex_type != previous_complex_type:
                previous_complex_type = complex_type
                writer.writerow([])  # insert blank lines between complex types

            writer.writerow([row] + [None]*(len(headers)-1))


def process_interfaces_spreadsheet(schema, spreadsheet, **_):
    """
    Reads a completed CSV spreadsheet and adds the attribute tags for each element back into the schema
    """
    with open(spreadsheet, 'r') as spreadsheet:
        reader = csv.DictReader(spreadsheet)

        for row in reader:
            if not row['Element']:
                continue
            typename, target = row['Element'].split(" - ")
            element = schema.xpath("//xsd:complexType[@name = '{}']//xsd:element[@name = '{}']".format(typename, target), namespaces=ns)
            if element:
                element = element[0]
            else:
                #print("typename=%s, target=%s not found" % (typename, target))
                continue

            # remove old mdl keys
            for old_key in element.attrib.keys():
                if old_key.startswith("{"+ns["mdl"]+"}"):
                    del element.attrib[old_key]

            attribs = defaultdict(list)
            for key, value in row.items():
                if key != 'Element' and value != '':
                    attribs[value].append(key)

            for key, value in attribs.items():
                element.set("{{{}}}{}".format(ns['mdl'], key), " ".join(value))

    print(etree.tounicode(schema))


def export_interfaces_spreadsheet(schema, spreadsheet, **_):
    """
    Exports a new spreadsheet based on the schema attribute tags
    """
    elements = schema.xpath("//xsd:element[ancestor::xsd:complexType]", namespaces=ns)

    headers = ["Element"] + interfaces

    with open(spreadsheet, 'w', newline='') as spreadsheet:
        writer = csv.DictWriter(spreadsheet, fieldnames=headers)
        writer.writeheader()

        previous_complex_type = elements[0].xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
        for element in elements:
            complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
            row = {"Element": complex_type + " - " + element.get('name')}

            for key, value in element.attrib.items():
                if key.startswith("{"+ns["mdl"]+"}"):
                    field = key[len("{"+ns["mdl"]+"}"):]
                    for component in value.split(" "):
                        row[component] = field

            if complex_type != previous_complex_type:
                previous_complex_type = complex_type
                writer.writerow({})  # insert blank lines between complex types

            writer.writerow(row)


def build_component_spreadsheet(component, schema, spreadsheet, **_):
    """
    Exports a new spreadsheet based on the schema attribute tags
    """
    elements = schema.xpath("//xsd:element[ancestor::xsd:complexType]", namespaces=ns)
    headers = ["Element", "Create", "Structure", "Config", "Export", "Negotiate", "Choice", "N/A"]

    with open(spreadsheet, 'w', newline='') as spreadsheet:
        writer = csv.DictWriter(spreadsheet, fieldnames=headers)
        writer.writeheader()

        previous_complex_type = elements[0].xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
        for element in elements:
            tags = []
            for key, value in element.attrib.items():
                if any(interface in value for interface in components[component]):
                    tags.append(key[len("{"+ns["mdl"]+"}"):])

            complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
            if complex_type != previous_complex_type:
                previous_complex_type = complex_type
                #writer.writerow({})  # insert blank lines between complex types

            row = OrderedDict()
            row["Element"] = complex_type + " - " + element.get('name')
            if tags:
                if "create" in tags: row["Create"] = "X"
                else: row["Create"] = ""
                if "structure" in tags: row["Structure"] = "X"
                else: row["Structure"] = ""
                if "config" in tags: row["Config"] = "X"
                else: row["Config"] = ""
                if "export" in tags: row["Export"] = "X"
                else: row["Export"] = ""
                if "negotiate" in tags: row["Negotiate"] = "X"
                else: row["Negotiate"] = ""
                if "choice" in tags: row["Choice"] = "X"
                else: row["Choice"] = ""
                row["N/A"] = ""
            else:
                row["N/A"] = "X"

            writer.writerow(row)


def build_chapter_spreadsheet(chapter, schema, spreadsheet, **_):
    """
    Exports a new spreadsheet based on the schema attribute tags
    """
    elements = schema.xpath("//xsd:element[ancestor::xsd:complexType]", namespaces=ns)

    if chapter in chapters.keys():
        headers = OrderedDict({"Section": None})

        # get interfaces for all chapter sections
        ch_ifs = list(itertools.chain(*[v for k,v in chapters[chapter].items()]))

        # build headers
        for element in elements:
            for key, value in element.attrib.items():
                if any(interface in value for interface in ch_ifs):
                    complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
                    #headers.append(complex_type + " - " + element.get('name'))
                    headers[complex_type] = None

        with open(spreadsheet, 'w', newline='') as spreadsheet:
            writer = csv.DictWriter(spreadsheet, fieldnames=list(headers.keys()))
            writer.writeheader()

            for k, v in chapters[chapter].items():
                row = OrderedDict()
                row["Section"] = k
                for element in elements:
                    for kk, vv in element.attrib.items():
                        if any(i in vv for i in v):
                            complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
                            #row[complex_type + " - " + element.get('name')] = "X"
                            row[complex_type] = "X"
                writer.writerow(row)
    else:
        headers = ["Element"]
        headers = (headers + list(ch22.keys()) + list(ch24.keys()) + list(ch25.keys()) +
                   list(ch26.keys()) + list(ch27.keys()) + list(ch28.keys()))
        #print(headers)

        # merge all sections to a single dict
        sections = {**ch22, **ch24, **ch25, **ch26, **ch27, **ch28}

        with open(spreadsheet, 'w', newline='') as spreadsheet:
            writer = csv.DictWriter(spreadsheet, fieldnames=headers)
            writer.writeheader()

            previous_complex_type = elements[0].xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
            for element in elements:
                row = OrderedDict()
                complex_type = element.xpath("ancestor::xsd:complexType/@name", namespaces=ns)[0]
                if complex_type != previous_complex_type:
                    previous_complex_type = complex_type
                    writer.writerow({})  # insert blank lines between complex types
                row["Element"] = complex_type + " - " + element.get('name')

                # get all interfaces for this element
                tag_ifs = list(itertools.chain(*[v.split() for k,v in element.attrib.items() if "MDL" in k]))
                #print(tag_ifs)

                for section, section_ifs in sections.items():
                    if any(interface in section_ifs for interface in tag_ifs):
                        row[section] = "X"
                writer.writerow(row)


if __name__ == "__main__":
    dispatch = {"create": create_spreadsheet,
                "process": process_interfaces_spreadsheet,
                "export": export_interfaces_spreadsheet,
                "build-dev": build_component_spreadsheet,
                "build-ch": build_chapter_spreadsheet}

    parser = argparse.ArgumentParser(description="tool for manipulating spreadsheets for the MDL schema")
    subparsers = parser.add_subparsers(help="function", dest='command')

    create = subparsers.add_parser("create", help="create empty spreadsheet from target schema with columns specified by type")
    create.add_argument("type", help="type of spreadsheet (interfaces, chapters, etc.)")
    create.add_argument("schema", help="target schema")
    create.add_argument("spreadsheet", help="location of generated spreadsheet")

    process = subparsers.add_parser("process", help="process spreadsheet and add attributes to the target schema")
    process.add_argument("schema", help="target schema")
    process.add_argument("spreadsheet", help="location of generated spreadsheet")

    export = subparsers.add_parser("export", help="export attributes of the target schema into a new spreadsheet")
    export.add_argument("schema", help="target schema")
    export.add_argument("spreadsheet", help="location of generated spreadsheet")

    build_dev = subparsers.add_parser("build-dev", help="build component spreadsheet from attributes in the target schema")
    build_dev.add_argument("component", help="component name")
    build_dev.add_argument("schema", help="target schema")
    build_dev.add_argument("spreadsheet", help="location of generated spreadsheet")

    build_ch = subparsers.add_parser("build-ch", help="build chapter spreadsheet from attributes in the target schema")
    build_ch.add_argument("chapter", help="chapter number")
    build_ch.add_argument("schema", help="target schema")
    build_ch.add_argument("spreadsheet", help="location of generated spreadsheet")

    args = parser.parse_args()

    if 'schema' in args:
        args.schema = etree.parse(args.schema)
        dispatch[args.command](**vars(args))
    else:
        parser.print_usage()