/*****************************************************************************
 *                  TmNS Source Selector Common Header File                  *
 *                                                                           *
 *                                                                           *
 *  Header file containing common functions for the TmNS Source Selector     *
 *  applications.  This includes both TSS Client and TSS Server reference    *
 *  applications.                                                            *
 *                                                                           *
 *  This code is not designed to be thread safe.                             *
 *                                                                           *
 *  Date: April 27, 2017                                                     *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/select.h>
#include <sys/time.h>
#include <arpa/inet.h> 
#include <netpacket/packet.h>
#include <net/if_arp.h>
#include <linux/if.h>
#include <linux/if_ether.h>
#include <resolv.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <malloc.h>
#include <stdint.h>
#include <netinet/tcp.h>


/*---------------------------------------------------------------------*/
/*--- CONSTANTS and MACROS                                          ---*/
/*---------------------------------------------------------------------*/

#define POLY 0x82608EDB

#define PERROR(x) do { perror(x); exit(__LINE__); } while (0)
#define PERROR_CONTINUE(x) do { perror("(trying to continue) " x);} while (0)
#define ERROR(x, args...) do { fprintf(stderr,"ERROR: " x, ## args); exit(__LINE__); } while (0)


/*---------------------------------------------------------------------*/
/*--- LOCAL STRUCT DEFINITIONS                                      ---*/
/*---------------------------------------------------------------------*/

typedef struct __attribute__((__packed__)) tss_msg_struct
{
    uint16_t len;
    uint32_t crc;
    unsigned char buffer[1024*4];
} TSS_MSG;


/*---------------------------------------------------------------------*/
/*--- InitServerCTX - initialize SSL server and create context      ---*/
/*---------------------------------------------------------------------*/
static SSL_CTX* InitServerCTX(void)
{   
    const SSL_METHOD *method;
    SSL_CTX *ctx;

    OpenSSL_add_all_algorithms();           /* load & register all cryptos, etc. */
    SSL_load_error_strings();               /* load all error messages */
    method = SSLv23_server_method();        /* create new server-method instance */
    ctx = SSL_CTX_new(method);              /* create new context from method */
    if ( ctx == NULL )
    {
        ERR_print_errors_fp(stderr);
        abort();
    }
    return ctx;
}


/*---------------------------------------------------------------------*/
/*--- InitCTX - initialize the SSL engine.                          ---*/
/*---------------------------------------------------------------------*/
SSL_CTX* InitCTX(void)
{
    const SSL_METHOD *method;
    SSL_CTX *ctx;

    OpenSSL_add_all_algorithms();           /* Load cryptos, et.al. */
    SSL_load_error_strings();               /* Bring in and register error messages */
    method = SSLv23_client_method();        /* Create new client-method instance */
    ctx = SSL_CTX_new(method);              /* Create new context */
    if ( ctx == NULL )
    {
        ERR_print_errors_fp(stderr);
        abort();
    }
    return ctx;
}


/*---------------------------------------------------------------------*/
/*--- LoadCertificates - load from files.                           ---*/
/*---------------------------------------------------------------------*/
static void LoadCertificates(SSL_CTX* ctx, char* CertFile, char* KeyFile)
{
	/* set the local certificate from CertFile */
    if ( SSL_CTX_use_certificate_file(ctx, CertFile, SSL_FILETYPE_PEM) <= 0 )
    {
        ERR_print_errors_fp(stderr);
        abort();
    }
    /* set the private key from KeyFile (may be the same as CertFile) */
    if ( SSL_CTX_use_PrivateKey_file(ctx, KeyFile, SSL_FILETYPE_PEM) <= 0 )
    {
        ERR_print_errors_fp(stderr);
        abort();
    }
    /* verify private key */
    if ( !SSL_CTX_check_private_key(ctx) )
    {
        fprintf(stderr, "Private key does not match the public certificate\n");
        abort();
    }
}


/*---------------------------------------------------------------------*/
/*--- ShowCerts - print out certificates.                           ---*/
/*---------------------------------------------------------------------*/
static void ShowCerts(SSL* ssl)
{   
    X509 *cert;
    char *line;

    cert = SSL_get_peer_certificate(ssl);	/* Get certificates (if available) */
    if ( cert != NULL )
    {
        printf("Server certificates:\n");
        line = X509_NAME_oneline(X509_get_subject_name(cert), 0, 0);
        printf("Subject: %s\n", line);
        free(line);
        line = X509_NAME_oneline(X509_get_issuer_name(cert), 0, 0);
        printf("Issuer: %s\n", line);
        free(line);
        X509_free(cert);
    }
    else
        printf("No certificates.\n");
}


/*---------------------------------------------------------------------*/
/*--- cwrite - write socket and check for error                     ---*/
/*---------------------------------------------------------------------*/
static inline int cwrite(SSL *ssl, int fd, unsigned char *buf, int n)
{
    int nwrite;

    if((nwrite=SSL_write(ssl, (void*)buf, n))<0) PERROR("Writing data");
    return nwrite;
}


/*---------------------------------------------------------------------*/
/*--- cread - read socket and check for error                       ---*/
/*---------------------------------------------------------------------*/
static inline int cread(SSL *ssl, int fd, unsigned char *buf, int n)
{  
    int nread;

    if((nread=SSL_read(ssl, (void*)buf, n))<0) PERROR("SSL_read()");
    return nread;
}


/*---------------------------------------------------------------------*/
/*--- bio_read - blocking I/O read of socket                        ---*/
/*---------------------------------------------------------------------*/
static inline int bio_read(SSL *ssl, int fd, unsigned char *buf, int n)
{
    int nread;
    int left = n;
    int grace = 2;

    while(left > 0)
    {
        nread = cread(ssl, fd, (void*)buf, left);
        if (nread == 0)
        {
            if (grace==0) PERROR("Forever lost in bio-read");
            else {sleep(1); grace--;}
        }
        else
        {
            left -= nread;
            buf += nread;
        }
    }
    return n;
}


/*---------------------------------------------------------------------*/
/*--- get_crcByte - perform byte calculations for CRC process       ---*/
/*---------------------------------------------------------------------*/
static inline uint32_t get_crcByte(int input)
{
    uint32_t val = input;
    int i;
    
    for (i=0; i<8; i++)
    {
        if (val & 1)
            val = (val >> 1) ^ POLY;
        else val >>= 1;
    }
    
    return val;
}


/*---------------------------------------------------------------------*/
/*--- get_crc32 - calculate the 32-bit CRC of the provided buffer   ---*/
/*---------------------------------------------------------------------*/
static inline uint32_t get_crc32(unsigned char *data, int sz)
{
    uint32_t remainder, t1, t2;
    int bytes;

    remainder = 0;
    
    for (bytes = 0; bytes < sz; bytes++)
    {
        t1 = (remainder >> 8) & 0x00FFFFFFL;
        t2 = get_crcByte(((int)remainder^(*(data+bytes)))&0xFF);
        remainder = t1^t2;
    }

    for (bytes = 0; bytes < sizeof(remainder); bytes++)
    {
        t1 = (remainder >> 8) & 0x00FFFFFFL;
        t2 = get_crcByte(((int)remainder)&0xFF);
        remainder = t1^t2;
    }

    return remainder;
}
