#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <jansson.h>
#include <csv.h>

struct counts {
    long unsigned fields;
    long unsigned rows;
};


struct csv_record
{
    char type[1024];
    char name[1024];
    char parent_name[1024];
    char position[1024];
    char resource_urn[1024];
    char oid[1024];
    char resource_syntax[1024];
    char access[1024];
    char default_val[1024];
    char table_index[1024];
    char date_intro[1024];
    char persistent[1024];
    char idempotency[1024];
    char description[1024];
    char comment[1024];
};

union record {
    struct csv_record records[1024];
    char v[1024][15][1024];
};

int rec_index = 0;
int row = 0, column = 0;
union record values;
FILE *fp;
struct counts c = {0, 0};

// functions used for csv parsing
void cb1 (void *s, size_t len, void *data);
void cb2 (int c, void *data);
static int is_term(unsigned char c);
static int is_space(unsigned char c);
void stripEndingQuotes();
void replaceColonsInURN();

// functions used to create JSON-formatted file
void addInitialSkeleton(json_t *root, json_t *paths_obj, json_t *defs_obj);
void addPath(json_t *paths_obj, const char *path, const char *oid_name, int put);
void addTablePath(json_t *paths_obj, char *path, const char *oid_name,
    const char *entry_name);
void addTableEntryPath(json_t *paths_obj, char *path, const char *oid_name,
    const char *entry_name);
void addScalarDefinition(json_t *defs_obj, const char *entry_name,
    const char *entry_desc, const char *oid_name, const char *oid_type,
    const char *oid_desc, int read_only);
void addEntryDefinition(json_t *defs_obj, const char *entry_name,
    const char *entry_desc, const char *oid_name, const char *oid_type,
    const char *oid_desc);
void addErrorDefinitions(json_t *defs_obj);
void addErrors(json_t *resp_obj);
void getIndexForThisTable(char *entry, char *index_name);
int isThisTableDynamic(char *entry);

int main(int argc, char *argv[])
{
    char *s = NULL;
    struct csv_parser p;
    char buf[1024];
    size_t bytes_read;
    unsigned char options = 0;
    int i;
    json_t *root = json_object();
    json_t *paths_obj = json_object();
    json_t *defs_obj = json_object();

    if (argc < 3) {
        fprintf(stderr, "Usage: ./csv2swag.exe <input csv file> <output json file>\n\n");
        exit(-1);
    }

    if (csv_init(&p, options) != 0) {
        fprintf(stderr, "Failed to initialize csv parser\n");
        exit(EXIT_FAILURE);
    }

    csv_set_space_func(&p, is_space);
    csv_set_term_func(&p, is_term);

    options = CSV_STRICT;
    csv_set_opts(&p, options);

    fp = fopen(argv[1], "r");
    if (!fp) {
        fprintf(stderr, "Failed to open %s: %s\n", argv[1], strerror(errno));
    }

    while ((bytes_read=fread(buf, 1, 1024, fp)) > 0) {
        if (csv_parse(&p, buf, bytes_read, cb1, cb2, &c) != bytes_read) {
            fprintf(stderr, "Error while parsing file: %s\n", csv_strerror(csv_error(&p)));
        }
    }

    csv_fini(&p, cb1, cb2, &c);

    if (ferror(fp)) {
        fprintf(stderr, "Error while reading file %s\n", argv[1]);
        fclose(fp);
    }

    fclose(fp);

    // strip end quotes from each string
    stripEndingQuotes();

    // replace colons with slashes in URN value
    replaceColonsInURN();

    // start initial json structure
    addInitialSkeleton(root, paths_obj, defs_obj);

    // skip first row of headers
    for(i = 1; i < c.rows; i++) {
        // skip first character since it is a quotation mark
        if (strcmp(&values.records[i].type[1], "TC") == 0)
            continue;
        else if (strcmp(&values.records[i].type[1], "B") == 0)
            continue;
        else if (strcmp(&values.records[i].type[1], "I") == 0)
            continue;
        else if (strcmp(&values.records[i].type[1], "N") == 0)
            continue;
        else if (strcmp(&values.records[i].type[1], "S") == 0) {
          if (strcmp(&values.records[i].access[1], "read-only") == 0) {
              addPath(paths_obj, &values.records[i].resource_urn[0], &values.records[i].name[1], 0);
              addScalarDefinition(defs_obj, &values.records[i].name[1], &values.records[i].description[1],
                  &values.records[i].name[1], &values.records[i].resource_syntax[1], &values.records[i].description[1], 1);
          }
          else {
              addPath(paths_obj, &values.records[i].resource_urn[0], &values.records[i].name[1], 1);
              addScalarDefinition(defs_obj, &values.records[i].name[1], &values.records[i].description[1],
                  &values.records[i].name[1], &values.records[i].resource_syntax[1], &values.records[i].description[1], 0);
          }
        }
        else if (strcmp(&values.records[i].type[1], "T") == 0) {
            addTablePath(paths_obj, &values.records[i].resource_urn[0], &values.records[i].name[1], &values.records[i].resource_syntax[1]);
            // 2nd argument, path (resource_urn), has been modified by previous function
            addTableEntryPath(paths_obj, &values.records[i].resource_urn[0], &values.records[i].name[1], &values.records[i].resource_syntax[1]);
        }
        else if (strcmp(&values.records[i].type[1], "ts") == 0) {
            addEntryDefinition(defs_obj, &values.records[i].parent_name[1], &values.records[i].parent_name[1],
              &values.records[i].name[1], &values.records[i].resource_syntax[1], &values.records[i].description[1]);
            if ((strcmp(&values.records[i].access[1], "read-only") == 0) || (strcmp(&values.records[i].access[1], "not-accessible") == 0))
                addScalarDefinition(defs_obj, &values.records[i].name[1], &values.records[i].description[1],
                    &values.records[i].name[1], &values.records[i].resource_syntax[1], &values.records[i].description[1], 1);
            else
                addScalarDefinition(defs_obj, &values.records[i].name[1], &values.records[i].description[1],
                    &values.records[i].name[1], &values.records[i].resource_syntax[1], &values.records[i].description[1], 0);
        }
        else
            printf("Error - undefined: %s\n", values.records[i].name);
    }

    // add error repsonse definition
    addErrorDefinitions(defs_obj);

    // create the json structure
    s = json_dumps(root, JSON_INDENT(4) | JSON_PRESERVE_ORDER);

    // dump json to the user-specified output file
    fp = fopen(argv[2], "w");
    fputs(s, fp);
    fclose(fp);

    json_decref(root);
    return 0;
}

void addInitialSkeleton(json_t *root, json_t *paths_obj, json_t *defs_obj)
{
    json_t *info_obj = json_object();
    json_t *schemes_arr = json_array();

    json_object_set_new(root, "swagger", json_string("2.0"));

    json_object_set_new(root, "info", info_obj);
    json_object_set_new(info_obj, "version", json_string("0.9.1"));
    json_object_set_new(info_obj, "title", json_string("TmNS REST API"));
    json_object_set_new(info_obj, "description", json_string("REST API for the TmNS Management Resources"));

    json_object_set_new(root, "schemes", schemes_arr );
    json_array_append(schemes_arr, json_string("http"));

    json_object_set_new(root, "paths", paths_obj);

    json_object_set_new(root, "definitions", defs_obj);
}

void addPath(json_t *paths_obj, const char *path, const char *oid_name, int put)
{
    json_t *urn_obj, *get_obj, *resp_obj, *r200_obj, *r200_sch_obj, *parm_obj, *p_sch_obj;
    json_t *prod_arr, *parm_arr;
    char summ_str[256], op_str[256], desc_str[256], defs_str[256];

    sprintf(summ_str, "Retrieves the %s OID", oid_name);
    sprintf(op_str, "get_%s", oid_name);
    sprintf(desc_str, "Successfully retrieved the %s OID", oid_name);
    sprintf(defs_str, "#/definitions/%s", oid_name);

    urn_obj = json_object();
    get_obj = json_object();
    prod_arr = json_array();
    resp_obj = json_object();
    r200_obj = json_object();
    r200_sch_obj = json_object();

    json_object_set_new(paths_obj, path, urn_obj);
    json_object_set_new(urn_obj, "get", get_obj);
    json_object_set_new(get_obj, "summary", json_string(summ_str));
    json_object_set_new(get_obj, "operationId", json_string(op_str));
    json_object_set_new(get_obj, "produces", prod_arr);
    json_array_append(prod_arr, json_string("application/xml"));
    json_object_set_new(get_obj, "responses", resp_obj);

    json_object_set_new(resp_obj, "200", r200_obj);
    json_object_set_new(r200_obj, "description", json_string(desc_str));
    json_object_set_new(r200_obj, "schema", r200_sch_obj);
    json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

    addErrors(resp_obj);

    // put
    if (put) {
        sprintf(summ_str, "Sets the value of the %s OID", oid_name);
        sprintf(op_str, "set_%s", oid_name);
        sprintf(desc_str, "Successfully set the %s OID", oid_name);

        get_obj = json_object();
        prod_arr = json_array();
        resp_obj = json_object();
        parm_obj = json_object();
        p_sch_obj = json_object();
        r200_obj = json_object();
        r200_sch_obj = json_object();
        parm_arr = json_array();

        json_object_set_new(urn_obj, "put", get_obj);
        json_object_set_new(get_obj, "summary", json_string(summ_str));
        json_object_set_new(get_obj, "operationId", json_string(op_str));
        json_object_set_new(get_obj, "produces", prod_arr);
        json_array_append(prod_arr, json_string("application/xml"));

        json_object_set_new(get_obj, "parameters", parm_arr);
        json_array_append(parm_arr, parm_obj);
        json_object_set_new(parm_obj, "name", json_string("id"));
        json_object_set_new(parm_obj, "in", json_string("body"));
        json_object_set_new(parm_obj, "required", json_true());
        json_object_set_new(parm_obj, "schema", p_sch_obj);
        json_object_set_new(p_sch_obj, "$ref", json_string(defs_str));

        json_object_set_new(get_obj, "responses", resp_obj);
        json_object_set_new(resp_obj, "200", r200_obj);
        json_object_set_new(r200_obj, "description", json_string(desc_str));
        json_object_set_new(r200_obj, "schema", r200_sch_obj);
        json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

        addErrors(resp_obj);
    }
}

void addTablePath(json_t *paths_obj, char *path, const char *oid_name, const char *entry_name)
{
    json_t *urn_obj, *get_obj, *resp_obj,
           *r200_obj, *r200_sch_obj;
    json_t *prod_arr, *item_obj, *parm_arr, *parm_obj;
    char summ_str[256], op_str[256], desc_str[256], defs_str[256], entry[256];

    sprintf(summ_str, "Retrieves the entire %s Table", oid_name);
    sprintf(op_str, "get_%s", oid_name);
    sprintf(desc_str, "Successfully retrieved the %s Table", oid_name);

    sscanf(entry_name, "SEQUENCE OF %s", entry);
    entry[0] = tolower(entry[0]);
    sprintf(defs_str, "#/definitions/%s", entry);

    int dynamic = isThisTableDynamic(entry);

    urn_obj = json_object();
    get_obj = json_object();
    prod_arr = json_array();
    resp_obj = json_object();
    r200_obj = json_object();
    r200_sch_obj = json_object();
    item_obj = json_object();

    int last_index = strlen(path);
    path[last_index-2] = '\0';

    json_object_set_new(paths_obj, path, urn_obj);

    // get
    json_object_set_new(urn_obj, "get", get_obj);
    json_object_set_new(get_obj, "summary", json_string(summ_str));
    json_object_set_new(get_obj, "operationId", json_string(op_str));
    json_object_set_new(get_obj, "produces", prod_arr);
    json_array_append(prod_arr, json_string("application/xml"));
    json_object_set_new(get_obj, "responses", resp_obj);

    json_object_set_new(resp_obj, "200", r200_obj);
    json_object_set_new(r200_obj, "description", json_string(desc_str));
    json_object_set_new(r200_obj, "schema", r200_sch_obj);
    json_object_set_new(r200_sch_obj, "type", json_string("array"));
    json_object_set_new(r200_sch_obj, "items", item_obj);
    json_object_set_new(item_obj, "$ref", json_string(defs_str));

    addErrors(resp_obj);

    // post for table row creation
    if (dynamic == 1) {
        sprintf(summ_str, "Creates a row of the %s Table", oid_name);
        sprintf(op_str, "create_%sEntry", oid_name);
        sprintf(desc_str, "Successfully created a row of the %s Table", oid_name);

        get_obj = json_object();
        prod_arr = json_array();
        resp_obj = json_object();
        r200_obj = json_object();
        r200_sch_obj = json_object();
        parm_obj = json_object();
        parm_arr = json_array();

        json_object_set_new(urn_obj, "post", get_obj);
        json_object_set_new(get_obj, "summary", json_string(summ_str));
        json_object_set_new(get_obj, "operationId", json_string(op_str));
        json_object_set_new(get_obj, "produces", prod_arr);
        json_array_append(prod_arr, json_string("application/xml"));
        json_object_set_new(get_obj, "responses", resp_obj);
        json_object_set_new(get_obj, "parameters", parm_arr);

        json_object_set_new(resp_obj, "200", r200_obj);
        json_object_set_new(r200_obj, "description", json_string(desc_str));
        json_object_set_new(r200_obj, "schema", r200_sch_obj);
        json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

        addErrors(resp_obj);

        json_array_append(parm_arr, parm_obj);
        json_object_set_new(parm_obj, "name", json_string(entry));
        json_object_set_new(parm_obj, "in", json_string("body"));
        json_object_set_new(parm_obj, "required", json_true());
        json_object_set_new(parm_obj, "schema", r200_sch_obj);
        json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));
    }
}

void addTableEntryPath(json_t *paths_obj, char *path, const char *oid_name, const char *entry_name)
{
    json_t *urn_obj, *get_obj, *resp_obj, *r200_obj, *r200_sch_obj, *item_obj, *parm_obj, *parm2_obj;
    json_t *prod_arr, *parm_arr;
    char summ_str[256], op_str[256], desc_str[256], defs_str[256], entry[256], index_name[256];

    sprintf(summ_str, "Retrieves a row of the %s Table", oid_name);
    sprintf(op_str, "get_%sEntry", oid_name);
    sprintf(desc_str, "Successfully retrieved a row of the %s Table", oid_name);

    sscanf(entry_name, "SEQUENCE OF %s", entry);
    entry[0] = tolower(entry[0]);
    sprintf(defs_str, "#/definitions/%s", entry);

    urn_obj = json_object();
    get_obj = json_object();
    prod_arr = json_array();
    resp_obj = json_object();
    r200_obj = json_object();
    r200_sch_obj = json_object();
    parm_obj = json_object();
    parm_arr = json_array();
    item_obj = json_object();

    getIndexForThisTable(entry, index_name);

    int dynamic = isThisTableDynamic(entry);

    int last_index = strlen(path);
    path[last_index] = '/';
    path[last_index+1] = '{';
    strcpy(&path[last_index+2], index_name);
    path[last_index+2+strlen(index_name)] = '}';
    path[last_index+2+strlen(index_name)+1] = '\0';

    json_object_set_new(paths_obj, path, urn_obj);

    // get
    json_object_set_new(urn_obj, "get", get_obj);
    json_object_set_new(get_obj, "summary", json_string(summ_str));
    json_object_set_new(get_obj, "operationId", json_string(op_str));
    json_object_set_new(get_obj, "produces", prod_arr);
    json_array_append(prod_arr, json_string("application/xml"));
    json_object_set_new(get_obj, "responses", resp_obj);
    json_object_set_new(get_obj, "parameters", parm_arr);

    json_object_set_new(resp_obj, "200", r200_obj);
    json_object_set_new(r200_obj, "description", json_string(desc_str));
    json_object_set_new(r200_obj, "schema", r200_sch_obj);
    json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

    addErrors(resp_obj);

    json_array_append(parm_arr, parm_obj);
    json_object_set_new(parm_obj, "name", json_string(index_name));
    json_object_set_new(parm_obj, "in", json_string("path"));
    json_object_set_new(parm_obj, "required", json_true());
    json_object_set_new(parm_obj, "type", json_string("integer"));

    // put
    sprintf(summ_str, "Sets a row of the %s Table", oid_name);
    sprintf(op_str, "put_%sEntry", oid_name);
    sprintf(desc_str, "Successfully set a row of the %s Table", oid_name);

    get_obj = json_object();
    prod_arr = json_array();
    resp_obj = json_object();
    r200_obj = json_object();
    r200_sch_obj = json_object();
    parm_obj = json_object();
    parm2_obj = json_object();
    parm_arr = json_array();
    item_obj = json_object();

    json_object_set_new(urn_obj, "put", get_obj);
    json_object_set_new(get_obj, "summary", json_string(summ_str));
    json_object_set_new(get_obj, "operationId", json_string(op_str));
    json_object_set_new(get_obj, "produces", prod_arr);
    json_array_append(prod_arr, json_string("application/xml"));
    json_object_set_new(get_obj, "responses", resp_obj);
    json_object_set_new(get_obj, "parameters", parm_arr);

    json_object_set_new(resp_obj, "200", r200_obj);
    json_object_set_new(r200_obj, "description", json_string(desc_str));
    json_object_set_new(r200_obj, "schema", r200_sch_obj);
    json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

    addErrors(resp_obj);

    json_array_append(parm_arr, parm_obj);
    json_object_set_new(parm_obj, "name", json_string(index_name));
    json_object_set_new(parm_obj, "in", json_string("path"));
    json_object_set_new(parm_obj, "required", json_true());
    json_object_set_new(parm_obj, "type", json_string("integer"));

    json_array_append(parm_arr, parm2_obj);
    json_object_set_new(parm2_obj, "name", json_string(entry));
    json_object_set_new(parm2_obj, "in", json_string("body"));
    json_object_set_new(parm2_obj, "required", json_true());
    json_object_set_new(parm2_obj, "schema", r200_sch_obj);
    json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

    // delete table row
    if (dynamic == 1) {
        sprintf(summ_str, "Deletes a row of the %s Table", oid_name);
        sprintf(op_str, "delete_%sEntry", oid_name);
        sprintf(desc_str, "Successfully deleted a row of the %s Table", oid_name);

        get_obj = json_object();
        prod_arr = json_array();
        resp_obj = json_object();
        r200_obj = json_object();
        r200_sch_obj = json_object();
        parm_obj = json_object();
        parm_arr = json_array();
        item_obj = json_object();

        json_object_set_new(urn_obj, "delete", get_obj);
        json_object_set_new(get_obj, "summary", json_string(summ_str));
        json_object_set_new(get_obj, "operationId", json_string(op_str));
        json_object_set_new(get_obj, "produces", prod_arr);
        json_array_append(prod_arr, json_string("application/xml"));
        json_object_set_new(get_obj, "responses", resp_obj);
        json_object_set_new(get_obj, "parameters", parm_arr);

        json_object_set_new(resp_obj, "200", r200_obj);
        json_object_set_new(r200_obj, "description", json_string(desc_str));
        json_object_set_new(r200_obj, "schema", r200_sch_obj);
        json_object_set_new(r200_sch_obj, "$ref", json_string(defs_str));

        addErrors(resp_obj);

        json_array_append(parm_arr, parm_obj);
        json_object_set_new(parm_obj, "name", json_string(index_name));
        json_object_set_new(parm_obj, "in", json_string("path"));
        json_object_set_new(parm_obj, "required", json_true());
        json_object_set_new(parm_obj, "type", json_string("integer"));
    }
}

void addEntryDefinition(json_t *defs_obj, const char *entry_name,
    const char *entry_desc, const char *oid_name, const char *oid_type,
    const char *oid_desc)
{
    const char *key;
    json_t *value, *temp;
    void *iter;
    json_t *prop_obj, *scal_obj;
    char ref_path_str[1024];

    // get defintions object
    iter = json_object_iter_at(defs_obj, entry_name);

    if (iter == NULL) {
        json_t *oid_obj, *prop_obj, *scal_obj;

        oid_obj = json_object();
        prop_obj = json_object();
        scal_obj = json_object();

        json_object_set_new(defs_obj, entry_name, oid_obj);
        json_object_set_new(oid_obj, "type", json_string("object"));
        json_object_set_new(oid_obj, "description", json_string(entry_desc));
        json_object_set_new(oid_obj, "properties", prop_obj);

        iter = json_object_iter_at(defs_obj, entry_name);
    }

    temp = json_object_iter_value(iter);

    // get properties object within definition
    iter = json_object_iter_at(temp, "properties");
    temp = json_object_iter_value(iter);

    // add the new entries
    prop_obj = json_object();
    scal_obj = json_object();
    json_object_set_new(temp, oid_name, scal_obj);

    sprintf(ref_path_str, "#/definitions/%s/properties/%s", oid_name, oid_name);
    json_object_set_new(scal_obj, "$ref", json_string(ref_path_str));
}

void addScalarDefinition(json_t *defs_obj, const char *entry_name,
    const char *entry_desc, const char *oid_name, const char *oid_type,
    const char *oid_desc, int read_only)
{
    const char *key;
    json_t *value, *temp;
    void *iter;
    json_t *oid_obj, *prop_obj, *scal_obj;


    // get defintions object
    iter = json_object_iter_at(defs_obj, entry_name);

    if (iter == NULL) {
        json_t *oid_obj, *prop_obj, *scal_obj;

        oid_obj = json_object();
        prop_obj = json_object();
        scal_obj = json_object();

        json_object_set_new(defs_obj, entry_name, oid_obj);
        json_object_set_new(oid_obj, "type", json_string("object"));
        json_object_set_new(oid_obj, "description", json_string(entry_desc));
        json_object_set_new(oid_obj, "properties", prop_obj);

        iter = json_object_iter_at(defs_obj, entry_name);
    }

    temp = json_object_iter_value(iter);

    // get properties object within definition
    iter = json_object_iter_at(temp, "properties");
    temp = json_object_iter_value(iter);

    // add the new entries
    prop_obj = json_object();
    scal_obj = json_object();
    json_object_set_new(temp, oid_name, scal_obj);

    json_object_set_new(scal_obj, "type", json_string("string"));
    json_object_set_new(scal_obj, "format", json_string(oid_type));
    json_object_set_new(scal_obj, "description", json_string(oid_desc));
    if (read_only == 1)
        json_object_set_new(scal_obj, "readOnly", json_true());
}


void cb1 (void *s, size_t len, void *data)
{
    ((struct counts *)data)->fields++;
    csv_write(values.v[row][column], 1024, s, len);
    column++;
    column = column % 15;
}

void cb2 (int c, void *data)
{
    ((struct counts *)data)->rows++;
    row++;
}

static int is_space(unsigned char c) {
    if (c == CSV_SPACE || c == CSV_TAB) return 1;
    return 0;
}

static int is_term(unsigned char c) {
    if (c == CSV_CR || c == CSV_LF) return 1;
    return 0;
}

// Only stripping end quotes, first quote will be stripped through proper indexing
void stripEndingQuotes()
{
    int i, j;
    // skip first row of headers
    for(i = 1; i < c.rows; i++) {
        for(j = 0; j < 15; j++) {
            int last_index = strlen(values.v[i][j]);
            values.v[i][j][last_index-1] = '\0';
        }
    }
}

void replaceColonsInURN()
{
    int i, j;
    // skip first row of headers
    for(i = 1; i < c.rows; i++) {
        int last_index = strlen(values.records[i].resource_urn);
        values.records[i].resource_urn[0] = '/';
        for(j = 0; j < last_index; j++) {
            if (values.records[i].resource_urn[j] == ':')
                values.records[i].resource_urn[j] = '/';
        }
    }
}

void addErrorDefinitions(json_t *defs_obj)
{
    addScalarDefinition(defs_obj, "errorTooBig", "Result was too big to put into a single SNMP message", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorTooBig", "Result was too big to put into a single SNMP message", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorTooBig", "Result was too big to put into a single SNMP message", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorNoSuchName", "Unknown variable", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorNoSuchName", "Unknown variable", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorNoSuchName", "Unknown variable", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorBadValue", "Syntax or value error", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorBadValue", "Syntax or value error", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorBadValue", "Syntax or value error", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorReadOnly", "Operation tried to change value that was not allowed to change", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorReadOnly", "Operation tried to change value that was not allowed to change", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorReadOnly", "Operation tried to change value that was not allowed to change", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorGeneral", "General error", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorGeneral", "General error", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorGeneral", "General error", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorNoAccess", "Variable is not accessible", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorNoAccess", "Variable is not accessible", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorNoAccess", "Variable is not accessible", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorWrongType", "Type is inconsistent with the type required", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorWrongType", "Type is inconsistent with the type required", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorWrongType", "Type is inconsistent with the type required", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorWrongLength", "Length is inconsistent with the length required", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorWrongLength", "Length is inconsistent with the length required", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorWrongLength", "Length is inconsistent with the length required", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorWrongEncoding", "ASN.1 encoding is inconsistent with the ASN.1 tag", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorWrongEncoding", "ASN.1 encoding is inconsistent with the ASN.1 tag", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorWrongEncoding", "ASN.1 encoding is inconsistent with the ASN.1 tag", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorWrongValue", "Value cannot be assigned to the variable", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorWrongValue", "Value cannot be assigned to the variable", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorWrongValue", "Value cannot be assigned to the variable", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorNoCreation", "Variable could not be created", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorNoCreation", "Variable could not be created", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorNoCreation", "Variable could not be created", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorInconsistentValue", "Value is inconsistent with values of other managed objects", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorInconsistentValue", "Value is inconsistent with values of other managed objects", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorInconsistentValue", "Value is inconsistent with values of other managed objects", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorResourceUnavailable", "Resources required to complete operation is currently unavailable", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorResourceUnavailable", "Resources required to complete operation is currently unavailable", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorResourceUnavailable", "Resources required to complete operation is currently unavailable", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorCommitFailed", "No variables were updated", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorCommitFailed", "No variables were updated", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorCommitFailed", "No variables were updated", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorUndoFailed", "It was not possible to undo some variable assignments", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorUndoFailed", "It was not possible to undo some variable assignments", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorUndoFailed", "It was not possible to undo some variable assignments", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorAuthorization", "Authorization error occurred", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorAuthorization", "Authorization error occurred", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorAuthorization", "Authorization error occurred", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorNotWritable", "Could not modify an existing variable", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorNotWritable", "Could not modify an existing variable", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorNotWritable", "Could not modify an existing variable", "Message", "string", "A detailed error message.", 0);

    addScalarDefinition(defs_obj, "errorInconsistentName", "Could not create the variable because it is inconsistent with the values of other managed objects", "URI", "string", "The request's URI.", 0);
    addScalarDefinition(defs_obj, "errorInconsistentName", "Could not create the variable because it is inconsistent with the values of other managed objects", "Method", "string", "The request's HTTP method.", 0);
    addScalarDefinition(defs_obj, "errorInconsistentName", "Could not create the variable because it is inconsistent with the values of other managed objects", "Message", "string", "A detailed error message.", 0);
}

void addErrors(json_t *resp_obj)
{
    json_t *error_obj, *error_sch_obj;

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "413", error_obj);
    json_object_set_new(error_obj, "description", json_string("Too Big Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorTooBig"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "404", error_obj);
    json_object_set_new(error_obj, "description", json_string("No Such Name Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorNoSuchName"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "400", error_obj);
    json_object_set_new(error_obj, "description", json_string("Bad Value Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorBadValue"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "405", error_obj);
    json_object_set_new(error_obj, "description", json_string("Read Only Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorReadOnly"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "500", error_obj);
    json_object_set_new(error_obj, "description", json_string("General Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorGeneral"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "403", error_obj);
    json_object_set_new(error_obj, "description", json_string("No Access Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorNoAccess"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "490", error_obj);
    json_object_set_new(error_obj, "description", json_string("Wrong Type Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorWrongType"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "491", error_obj);
    json_object_set_new(error_obj, "description", json_string("Wrong Length Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorWrongLength"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "492", error_obj);
    json_object_set_new(error_obj, "description", json_string("Wrong Encoding Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorWrongEncoding"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "493", error_obj);
    json_object_set_new(error_obj, "description", json_string("Wrong Value Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorWrongValue"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "507", error_obj);
    json_object_set_new(error_obj, "description", json_string("No Creation Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorNoCreation"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "494", error_obj);
    json_object_set_new(error_obj, "description", json_string("Inconsistent Value Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorInconsistentValue"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "501", error_obj);
    json_object_set_new(error_obj, "description", json_string("Resource Unavailable Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorResourceUnavailable"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "590", error_obj);
    json_object_set_new(error_obj, "description", json_string("Commit Failed Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorCommitFailed"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "591", error_obj);
    json_object_set_new(error_obj, "description", json_string("Undo Failed Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorUndoFailed"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "401", error_obj);
    json_object_set_new(error_obj, "description", json_string("Authorization Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorAuthorization"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "592", error_obj);
    json_object_set_new(error_obj, "description", json_string("Not Writable Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorNotWritable"));

    error_obj = json_object();
    error_sch_obj = json_object();
    json_object_set_new(resp_obj, "593", error_obj);
    json_object_set_new(error_obj, "description", json_string("Inconsistent Name Error"));
    json_object_set_new(error_obj, "schema", error_sch_obj);
    json_object_set_new(error_sch_obj, "$ref", json_string("#/definitions/errorInconsistentName"));
}

void getIndexForThisTable(char *entry, char *index_name)
{
    int i;
    // skip first row of headers
    for(i = 1; i < c.rows; i++) {
        // skip first character since it is a quotation mark
        if (strcmp(&values.records[i].type[1], "ts") == 0) {
            if (strcmp(&values.records[i].parent_name[1], entry) == 0) {
                if (strcmp(&values.records[i].table_index[1], "1") == 0) {
                    strcpy(index_name, &values.records[i].name[1]);
                    break;
                }
            }
        }
    }
}

int isThisTableDynamic(char *entry)
{
    int i;
    // skip first row of headers
    for(i = 1; i < c.rows; i++) {
        // skip first character since it is a quotation mark
        if (strcmp(&values.records[i].type[1], "ts") == 0) {
            if (strcmp(&values.records[i].parent_name[1], entry) == 0) {
                if (strcmp(&values.records[i].resource_syntax[1], "RowStatus") == 0) {
                    return 1;
                }
            }
        }
    }
    return 0;
}
