#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <csv.h>


void stripEndingQuotes();
void generateTextualConvention(int i);
void generateBranch(int i);
void generateIdentity(int i);
void generateScalar(int i);
void generateTable(int i);
void generateTableScalar(int i);
void generateNotification(int i);
void copyFile(char *file_to_copy);
void prettyPrintEnums(char *old, char *new);
void prettyPrintObjects(char *old, char *new);

struct counts {
  long unsigned fields;
  long unsigned rows;
};

struct csv_record
{
    char type[1024];
    char name[1024];
    char parent_name[1024];
    char position[1024];
    char resource_urn[1024];
    char oid[1024];
    char resource_syntax[1024];
    char access[1024];
    char default_val[1024];
    char table_index[1024];
    char date_intro[1024];
    char persistent[1024];
    char idempotency[1024];
    char description[1024];
    char comment[1024];
};

int rec_index = 0;
struct csv_record records[1024];
int row = 0, column = 0;

char values[1024][15][1024];

void cb1 (void *s, size_t len, void *data) 
{ 
    ((struct counts *)data)->fields++; 
    csv_write(values[row][column], 1024, s, len); 
    column++;
    column = column % 15;
}

void cb2 (int c, void *data) 
{ 
    ((struct counts *)data)->rows++; 
    row++;
}

static int is_space(unsigned char c) {
  if (c == CSV_SPACE || c == CSV_TAB) return 1;
  return 0;
}

static int is_term(unsigned char c) {
  if (c == CSV_CR || c == CSV_LF) return 1;
  return 0;
}

FILE *fp;
struct counts c = {0, 0};

int main (int argc, char *argv[])
{
    struct csv_parser p;
    char buf[1024];
    size_t bytes_read;
    unsigned char options = 0;

    if (argc < 3) {
      fprintf(stderr, "Usage: %s <input csv file> <output mib file>\n\n", argv[0]);
      exit(-1);
    }

    if (csv_init(&p, options) != 0) {
        fprintf(stderr, "Failed to initialize csv parser\n");
        exit(EXIT_FAILURE);
    }

    csv_set_space_func(&p, is_space);
    csv_set_term_func(&p, is_term);

    options = CSV_STRICT;
    csv_set_opts(&p, options);

    //fp = fopen("MIB.csv", "r");
    fp = fopen(argv[1], "r");
    if (!fp) {
        fprintf(stderr, "Failed to open %s: %s\n", argv[1], strerror(errno));
    }

    while ((bytes_read=fread(buf, 1, 1024, fp)) > 0) {
        if (csv_parse(&p, buf, bytes_read, cb1, cb2, &c) != bytes_read) {
            fprintf(stderr, "Error while parsing file: %s\n", csv_strerror(csv_error(&p)));
        }
    }

    csv_fini(&p, cb1, cb2, &c);

    if (ferror(fp)) {
      fprintf(stderr, "Error while reading file %s\n", *argv);
      fclose(fp);
    }

    fclose(fp);
    printf("Found a total of %lu fields, across %lu rows\n", c.fields, c.rows);

    int i;

    fp = fopen(argv[2], "w");
    if (!fp) {
      fprintf(stderr, "Failed to open %s for writing: %s\n", argv[2], strerror(errno));
    }

    copyFile("templates/header.txt");

    stripEndingQuotes();

    // generate identity first
    // skip first row of headers
    for(i = 1; i < c.rows; i++)
    {
        // skip first character since it is a quotation mark
        if (strcmp(&values[i][0][1], "I") == 0)
        {
            generateIdentity(i);
            fprintf(fp, "\n");
        }
    }

    // next is textual conventions
    // skip first row of headers
    for(i = 1; i < c.rows; i++)
    {
        // skip first character since it is a quotation mark
        if (strcmp(&values[i][0][1], "TC") == 0)
        {
            generateTextualConvention(i);
            fprintf(fp, "\n");
        }
    }

    // next is branches
    // skip first row of headers
    for(i = 1; i < c.rows; i++)
    {
        if (strcmp(&values[i][0][1], "B") == 0)
        {
            generateBranch(i);
            fprintf(fp, "\n");
        }
    }

    // then the rest
    // skip first row of headers
    for(i = 1; i < c.rows; i++)
    {
        // skip first character since it is a quotation mark
        if (strcmp(&values[i][0][1], "TC") == 0)
            continue;
        else if (strcmp(&values[i][0][1], "B") == 0)
            continue;
        else if (strcmp(&values[i][0][1], "I") == 0)
            continue;
        else if (strcmp(&values[i][0][1], "S") == 0)
            generateScalar(i);
        else if (strcmp(&values[i][0][1], "T") == 0)
            generateTable(i);
        else if (strcmp(&values[i][0][1], "ts") == 0)
            generateTableScalar(i);
        else if (strcmp(&values[i][0][1], "N") == 0)
            generateNotification(i);
        else
            printf("Error - undefined: %s\n", values[i][1]);

        fprintf(fp, "\n");
    }

    copyFile("templates/footer.txt");

    csv_free(&p);
    return 0;;
}

// Only stripping end quotes, first quote will be stripped through proper indexing
void stripEndingQuotes()
{
    int i, j;
    // skip first row of headers
    for(i = 1; i < c.rows; i++)
    {
        for(j = 0; j < 15; j++)
        {
            int last_index = strlen(values[i][j]);
            values[i][j][last_index-1] = '\0';
        }
    }
}
 
void generateTextualConvention(int i)
{
    char new_syntax[1024];

    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    prettyPrintEnums(&values[i][6][1], new_syntax);

    fprintf(fp, "%s ::= TEXTUAL-CONVENTION\n", &values[i][1][1]);
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"%s\"\n", &values[i][13][1]);
    fprintf(fp, "\tSYNTAX  %s\n", new_syntax);
}

void generateBranch(int i)
{
    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    // TODO: fix whitespace formatting
    fprintf(fp, "%s        OBJECT IDENTIFIER ::= { %s %s }\n", &values[i][1][1], &values[i][2][1], &values[i][3][1]);
}

void generateIdentity(int i)
{
    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    fprintf(fp, "%s MODULE-IDENTITY\n", &values[i][1][1]);

    copyFile("templates/identity.txt");

    fprintf(fp, "::= { %s %s }\n",  &values[i][2][1],  &values[i][3][1]);
}

void generateScalar(int i)
{
    char defval[1024];
    char new_syntax[1024];

    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    prettyPrintEnums(&values[i][6][1], new_syntax);

    fprintf(fp, "%s  OBJECT-TYPE\n", &values[i][1][1]);
//    fprintf(fp, "\tSYNTAX     %s\n", &values[i][6][1]);
    fprintf(fp, "\tSYNTAX     %s\n", new_syntax);
    fprintf(fp, "\tMAX-ACCESS %s\n", &values[i][7][1]);
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"%s\"\n", &values[i][13][1]);

    // CSV conversion adds multiple quotation marks for some reason, check and limit output to two
    if (values[i][8][3] == '\"')
    {
        fprintf(fp, "\tDEFVAL  { \"\" }\n");
    }
    else if (values[i][8][2] == '\"')
    {
        sscanf(&values[i][8][3], "%[^\"]", defval);
        fprintf(fp, "\tDEFVAL  { \"%s\" }\n", defval);
    }
    else if (values[i][8][1] == '\0')
        ;  // do nothing
    else
    {
        fprintf(fp, "\tDEFVAL  { %s }\n", &values[i][8][1]);
    }

    fprintf(fp, "\t::= { %s %s }\n",  &values[i][2][1],  &values[i][3][1]);
}

void generateTable(int i)
{
    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    fprintf(fp, "%s  OBJECT-TYPE\n", &values[i][1][1]);
    fprintf(fp, "\tSYNTAX %s\n", &values[i][6][1]);
    fprintf(fp, "\tMAX-ACCESS %s\n", &values[i][7][1]);
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"%s\"\n", &values[i][13][1]);
    fprintf(fp, "\t::= { %s %s }\n\n",  &values[i][2][1],  &values[i][3][1]);

    // generate entry and SEQUENCE from implicit info

    // convert first character of Table SYNTAX to lowercase
    char entry_name[1024];
    char entry_name_lower[1024];

    sscanf(&values[i][6][1], "%*s %*s %s", entry_name);

    strcpy(entry_name_lower, entry_name);
    entry_name_lower[0] = tolower(entry_name[0]);

    fprintf(fp, "%s  OBJECT-TYPE\n", entry_name_lower);
    fprintf(fp, "\tSYNTAX  %s\n", entry_name);
    fprintf(fp, "\tMAX-ACCESS not-accessible\n");
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"A row in the %s.\"\n", &values[i][1][1]);
    fprintf(fp, "\tINDEX { ");

    // this is a simple, but probably inefficient way of doing this
    int j;
    int first = 1;
    for(j = 1; j < c.rows; j++)
    {
        // look for matching parent type
        if (strcmp(&values[j][2][1], entry_name_lower) == 0)
        {
            if (values[j][9][1] != '\0')
            {
                if (first == 0)
                    fprintf(fp, ",\n\t\t");
                first = 0;
                fprintf(fp, "%s", &values[j][1][1]);
            }
        }
    }
    fprintf(fp, " }\n");
    fprintf(fp, "\t::= { %s 1 }\n\n",  &values[i][1][1]);
    
    fprintf(fp, "%s ::= SEQUENCE {\n", entry_name);
    // this is a simple, but probably inefficient way of doing this
    first = 1;
    for(j = 1; j < c.rows; j++)
    {
        // look for matching parent type
        if (strcmp(&values[j][2][1], entry_name_lower) == 0)
        {
            char type[1024];
            if (first == 0)
                fprintf(fp, ",\n");
            first = 0;
            // do not grab the entire enumeration
            sscanf(&values[j][6][1], "%[^{(]", type);
            // remove trailling whitespace if necessary
            if (type[strlen(type)-1] == ' ')
                type[strlen(type)-1] = '\0';
            fprintf(fp, "\t%s\n\t\t%s", &values[j][1][1], type);
        }
    }
    fprintf(fp, "\n}\n");
}

void generateTableScalar(int i)
{
    char defval[1024];
    char new_syntax[1024];

    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    prettyPrintEnums(&values[i][6][1], new_syntax);

    fprintf(fp, "%s  OBJECT-TYPE\n", &values[i][1][1]);
//    fprintf(fp, "\tSYNTAX     %s\n", &values[i][6][1]);
    fprintf(fp, "\tSYNTAX     %s\n", new_syntax);
    fprintf(fp, "\tMAX-ACCESS %s\n", &values[i][7][1]);
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"%s\"\n", &values[i][13][1]);

    // CSV conversion adds multiple quotation marks for some reason, check and limit output to two
    if (values[i][8][3] == '\"')
    {
        fprintf(fp, "\tDEFVAL  { \"\" }\n");
    }
    else if (values[i][8][2] == '\"')
    {
        sscanf(&values[i][8][3], "%[^\"]", defval);
        fprintf(fp, "\tDEFVAL  { \"%s\" }\n", defval);
    }
    else if (values[i][8][1] == '\0')
        ;  // do nothing
    else
    {
        fprintf(fp, "\tDEFVAL  { %s }\n", &values[i][8][1]);
    }

    fprintf(fp, "\t::= { %s %s }\n",  &values[i][2][1],  &values[i][3][1]);
}

void generateNotification(int i)
{
    char new_syntax[1024];

    // add comment above if necessary
    if (values[i][14][1] != '\0')
        fprintf(fp, "-- %s\n\n",  &values[i][14][1]);

    prettyPrintObjects(&values[i][6][1], new_syntax);

    fprintf(fp, "%s NOTIFICATION-TYPE\n", &values[i][1][1]);
//    fprintf(fp, "\t%s\n", &values[i][6][1]);
    fprintf(fp, "\t%s\n", new_syntax);
    fprintf(fp, "\tSTATUS     current\n");
    fprintf(fp, "\tDESCRIPTION\n");
    fprintf(fp, "\t\t\"%s\"\n", &values[i][13][1]);
    fprintf(fp, "\t::= { %s %s }\n",  &values[i][2][1],  &values[i][3][1]);
}

void copyFile(char *file_to_copy)
{
    FILE *fp_tmp;
    char * line = NULL;
    size_t len = 0;
    ssize_t read;

    fp_tmp = fopen(file_to_copy, "r");
    if (fp_tmp == NULL)
        fprintf(stderr, "Failed to find file to copy: %s\n", file_to_copy);

    while ((read = getline(&line, &len, fp_tmp)) != -1) {
        fprintf(fp, "%s", line);
    }

    fclose(fp_tmp);
    if (line)
        free(line);
}

void prettyPrintEnums(char *old, char *new)
{
    int old_i = 0, new_i = 0;
    while(old[old_i] != '\0')
    {
        if ((old[old_i] == '{') || (old[old_i] == ','))
        {
            new[new_i++] = old[old_i];
            new[new_i++] = '\n';
            new[new_i++] = '\t';
            new[new_i++] = '\t';
        }
        else if (old[old_i] == '}')
        {
            new[new_i++] = '\n';
            new[new_i++] = '\t';
            new[new_i++] = old[old_i];
        }
        else 
            new[new_i++] = old[old_i];

        old_i++;
    }
    new[new_i] = '\0';
}

void prettyPrintObjects(char *old, char *new)
{
    int old_i = 0, new_i = 0;
    while(old[old_i] != '\0')
    {
        if (old[old_i] == ',')
        {
            new[new_i++] = old[old_i];
            new[new_i++] = '\n';
            new[new_i++] = '\t';
            new[new_i++] = '\t';
        }
        else 
            new[new_i++] = old[old_i];

        old_i++;
    }
    new[new_i] = '\0';
}
